/*
 * USB Host-to-Host Links
 * Copyright (C) 2000-2002 by David Brownell <dbrownell@users.sourceforge.net>
 * Copyright (C) 2002 Pavel Machek <pavel@ucw.cz>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * Modified usbnet.c with the changes for moschip usb ethernet adapter (MCS7830)
 *
 *-------------------------------------------------------------------------*/

#include <linux/config.h>
#include <linux/module.h>
#include <linux/kmod.h>
#include <linux/sched.h>
#include <linux/init.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/random.h>
#include <linux/ethtool.h>
//#include <linux/mii.h>
#include <linux/tqueue.h>
#include <asm/uaccess.h>
#include <asm/unaligned.h>

#include <linux/spinlock.h>
#include <linux/smp_lock.h>
#include <linux/blk.h>

#include "mcs7830.h"


// #define	DEBUG			// error path messages, extra info
// #define	VERBOSE			// more; success messages
// #define	REALLY_QUEUE

#if !defined (DEBUG) && defined (CONFIG_USB_DEBUG)
#   define DEBUG
#endif
#include <linux/usb.h>

/* in 2.5 these standard usb ops take mem_flags */
#define ALLOC_URB(n,flags)	usb_alloc_urb(n)
#define SUBMIT_URB(u,flags)	usb_submit_urb(u)

/* and these got renamed (may move to usb.h) */
#define usb_get_dev		usb_inc_dev_use
#define usb_put_dev		usb_dec_dev_use


/* minidrivers _could_ be individually configured */


#define DRIVER_VERSION		"5-sep-2008"


#define NEGOTIATE_MODE 0
//Mode to select negotiation mode
 //0 Auto negotiation
 //1 10Half
 //2 10Full
 //3 100Half
 //4 100Full 
 
#define  PACKET_STATUS  1
#define  IP_ALIGN	1
 
 
/*-------------------------------------------------------------------------*/

/*
 * Nineteen USB 1.1 max size bulk transactions per frame (ms), max.
 * Several dozen bytes of IPv4 data can fit in two such transactions.
 * One maximum size Ethernet packet takes twenty four of them.
 * For high speed, each frame comfortably fits almost 36 max size
 * Ethernet packets (so queues should be bigger).
 */
#ifdef REALLY_QUEUE
#define	RX_QLEN		4
#define	TX_QLEN		4
#else
#define	RX_QLEN		1
#define	TX_QLEN		1
#endif

// packets are always ethernet inside
// ... except they can be bigger (limit of 64K with NetChip framing)
#define MIN_PACKET	sizeof(struct ethhdr)
#define MAX_PACKET	32768

// reawaken network queue this soon after stopping; else watchdog barks
#define TX_TIMEOUT_JIFFIES	(5*HZ)

// for vendor-specific control operations
#define	CONTROL_TIMEOUT_MS	(500)			/* msec */
#define CONTROL_TIMEOUT_JIFFIES ((CONTROL_TIMEOUT_MS * HZ)/1000)

// between wakeups
#define UNLINK_TIMEOUT_JIFFIES ((3  /*ms*/ * HZ)/1000)

/*-------------------------------------------------------------------------*/
// number of times the probe is failed in one go
int no_of_times_probe_failed=0;


// list of all devices we manage
static DECLARE_MUTEX (mcs7830_mutex);
static LIST_HEAD (mcs7830_list);

// randomly generated ethernet address
static u8	node_id [ETH_ALEN];


#pragma pack(4)
// state we keep for each device we handle

struct mcs7830 {
        //Rev.C change
        __u8                                                   ptrPauseThreshold;
	__u8     SpeedDuplex;  // New reg value for speed/duplex
	__u8     ForceDuplex;  // 1 for half-duplex, 2 for full duplex
        __u8     ForceSpeed;   // 10 for 10Mbps, 100 for 100 Mbps
        __u8           AiPermanentNodeAddress[ETHERNET_ADDRESS_LENGTH];
        __u8           bAutoNegStarted;
        __u8           HifReg15;
        __u8           ByteValue [10];
        __u8           checkval2[10];//added for checking removing later
        __u8          TempPhyControlReg[2];
        __u8          PhyControlReg[2];
        __u16          WordValue;
        //__u16          PhyControlReg;
        __u16          PhyAutoNegLinkPartnerReg;
        __u16          PhyChipStatusReg;
        //__u16          PhyAutoNegAdvReg;
        //__u16          TempPhyControlReg;

        __u16          DeviceReleaseNumber;


        //__u32          CurrentPacketFilter;
        //__u32                                                  ptrPauseTime;

        __u32                                                  ptrFpgaVersion;
        //__u32                                                  ptrRxErrorCount;

	// housekeeping
	struct usb_device	*udev;
	struct driver_info	*driver_info;
	struct semaphore	mutex;
	struct list_head	dev_list;
	wait_queue_head_t	*wait;

	// i/o info: pipes etc
	unsigned		in, out;
	unsigned		maxpacket;
	//struct timer_list	delay;

	// protocol/interface state
	struct net_device	net;
	struct net_device_stats	stats;
	unsigned int			msg_level;
	//struct mii_if_info	mii;


	// various kinds of pending driver work
	struct sk_buff_head	rxq;
	struct sk_buff_head	txq;
	struct sk_buff_head	done;
	struct tasklet_struct	bh;

	struct tq_struct	kevent;
	unsigned long		flags;
#		define EVENT_TX_HALT	0
#		define EVENT_RX_HALT	1
#		define EVENT_RX_MEMORY	2
// usage for the vendor command MOSCHIP change !!!!
	unsigned int 	is_7830_connected;//is it 7830?	


};


// device-specific info used by the driver
struct driver_info {
	/* for new devices, use the descriptor-reading code instead */
	unsigned int		in;		/* rx endpoint */
	unsigned int		out;		/* tx endpoint */
	unsigned int		epsize;

	unsigned long	data;		/* Misc driver specific data */
	int		flags;
	char		*description;

#define FLAG_FRAMING_NC	0x0001		/* guard against device dropouts */ 
#define FLAG_FRAMING_GL	0x0002		/* genelink batches packets */
#define FLAG_FRAMING_Z	0x0004		/* zaurus adds a trailer */
#define FLAG_NO_SETINT	0x0010		/* device can't set_interface() */
#define FLAG_ETHER	0x0020		/* maybe use "eth%d" names */

	/* init device ... can sleep, or cause probe() failure */
	int	(*bind)(struct mcs7830 *, struct usb_device *);

	/* reset device ... can sleep */
	int	(*reset)(struct mcs7830 *);

	/* see if peer is connected ... can sleep */
	int	(*check_connect)(struct mcs7830 *);

	/* fixup rx packet (strip framing) */
	int	(*rx_fixup)(struct mcs7830 *dev, struct sk_buff *skb);

	/* fixup tx packet (add framing) */
	struct sk_buff	*(*tx_fixup)(struct mcs7830 *dev,
				struct sk_buff *skb, int flags);

	// FIXME -- also an interrupt mechanism
	// useful for at least PL2301/2302 and GL620USB-A

};

// we record the state for each of our queued skbs
enum skb_state {
	illegal = 0,
	tx_start, tx_done,
	rx_start, rx_done, rx_cleanup
};

struct skb_data {	// skb->cb is one of these
	size_t			length;
	struct urb		*urb;
	struct mcs7830		*dev;
	enum skb_state		state;
};

static const char driver_name [] = "mcs_7830";

#pragma pack()
/* use ethtool to change the level for any given device */
static int msg_level = 1;
MODULE_PARM (msg_level, "i");
MODULE_PARM_DESC (msg_level, "Initial message level (default = 1)");




#define	mutex_lock(x)	down(x)
#define	mutex_unlock(x)	up(x)

#define	RUN_CONTEXT (in_irq () ? "in_irq" \
			: (in_interrupt () ? "in_interrupt" : "can sleep"))

//static struct ethtool_ops mcs7830_ethtool_ops;

/* mostly for PDA style devices, which are always present */
static int always_connected (struct mcs7830 *dev)
{
	return 0;
}

/*-------------------------------------------------------------------------*/

/* handles CDC Ethernet and many other network "bulk data" interfaces */
static int
get_endpoints (struct mcs7830 *dev, struct usb_interface *intf)
{
	int				tmp;
	struct usb_interface_descriptor	*alt;
	struct usb_endpoint_descriptor	*in, *out;

	for (tmp = 0; tmp < intf->max_altsetting; tmp++) {
		unsigned	ep;

		in = out = 0;
		alt = intf->altsetting + tmp;

		/* take the first altsetting with in-bulk + out-bulk;
		 * ignore other endpoints and altsetttings.
		 */
		for (ep = 0; ep < alt->bNumEndpoints; ep++) {
			struct usb_endpoint_descriptor	*e;

			e = alt->endpoint + ep;
			if (e->bmAttributes != USB_ENDPOINT_XFER_BULK)
				continue;
			if (e->bEndpointAddress & USB_DIR_IN) {
				if (!in)
					in = e;
			} else {
				if (!out)
					out = e;
			}
			if (in && out)
				goto found;
		}
	}
	return -EINVAL;

found:
	if (alt->bAlternateSetting != 0
			|| !(dev->driver_info->flags & FLAG_NO_SETINT)) {
		tmp = usb_set_interface (dev->udev, alt->bInterfaceNumber,
				alt->bAlternateSetting);
		if (tmp < 0)
			return tmp;
	}
	
	dev->in = usb_rcvbulkpipe (dev->udev,
			in->bEndpointAddress & USB_ENDPOINT_NUMBER_MASK);
	dev->out = usb_sndbulkpipe (dev->udev,
			out->bEndpointAddress & USB_ENDPOINT_NUMBER_MASK);
	dev->maxpacket = usb_maxpacket (dev->udev, dev->out, 1);
	return 0;
}



/*-------------------------------------------------------------------------*/





#ifdef DEBUG
#define devdbg(mcs7830, fmt, arg...) \
	printk(KERN_DEBUG "%s: " fmt "\n" , (mcs7830)->net.name , ## arg)
#else
#define devdbg(mcs7830, fmt, arg...) do {} while(0)
#endif

#define devinfo(mcs7830, fmt, arg...) \
	do { if ((mcs7830)->msg_level >= 1) \
	printk(KERN_INFO "%s: " fmt "\n" , (mcs7830)->net.name , ## arg); \
	} while (0)



#if 1

/**************************MOSCHIP CODE*********************/

#define CONFIG_USB_MOSCHIP

/*
 * Debug related defines
 */

/* 1: Enables the debugging -- 0: Disable the debugging */

#define MOS_DEBUG       0

#if MOS_DEBUG
        //static int debug = 0;
        #define DPRINTK(fmt, args...) printk( "%s: " fmt, __FUNCTION__ , ## args)

#else
        //static int debug = 0;
        #define DPRINTK(fmt, args...)

#endif



/*-------------------------------------------------------------------------*/

// MOSCHIP functions start

#define MSECS_TO_JIFFIES(ms) (((ms)*HZ+999)/1000) //Original Line
//#define MS  (2000)//(500)
//#define MSECS_TO_JIFFIES(ms) ((MS* HZ)/1000)

#define NETH_USB 4

#define WAIT_FOR_EVER   (HZ * 0 ) /* timeout urb is wait for ever*/
#define MOS_WDR_TIMEOUT (HZ * 5 ) /* default urb timeout */


#define MCS_VENDOR_ID 0x9710    /* Vendor Id of Moschip MCS7830 */
#define MCS_PRODUCT_ID 0x7830   /* Product Id  */

#define MOS_WRITE       0x0D
#define MOS_READ        0x0E
// mii.h values coresponding to return values of mdio_read
#define BIT_13      0x2000
#define BIT_14      0x4000
#define BIT_15      0x8000
#define BIT_16      0x00010000
#define BIT_17      0x00020000
#define BIT_18      0x00040000
#define BIT_19      0x00080000
#define BIT_20      0x00100000
#define BIT_21      0x00200000
#define BIT_22      0x00400000
#define BIT_23      0x00800000
#define BIT_24      0x01000000
#define LINKOK      0x0004


/* setting and get register values */
#ifdef comment
static int GetVendorCommandPassive(struct mcs7830 *mcs, __u16 reg_index,__u16 length, __u8 * data);
static int SetVendorCommandPassive(struct mcs7830 *mcs, __u16 reg_index,__u16 length, __u8 * data);
static int mcs7830_find_endpoints(struct mcs7830 *mcs, struct usb_endpoint_descriptor *ep, int epnum);
static int MosUsbGetPermanentAddress(struct mcs7830 *mcs);
static int AutoNegChangemode(struct mcs7830 *mcs,int ptrUserPhyMode);
static int ANGInitializeDev(struct mcs7830 *mcs);


static int ReadPhyRegisterPassive(struct mcs7830 *mcs,__u8 phy_reg_index,__u16 *phy_data);
static int WritePhyRegisterPassive(struct mcs7830 *mcs,__u8 PhyRegIndex,__u16 *PhyRegValue);
#endif


static void mcs7830_disconnect (struct usb_device *udev, void *ptr);
#ifdef CONFIG_USB_MOSCHIP
static const struct driver_info moschip_info = {
        .description =  "MOSCHIP 7830 usb-NET adapter",
        .check_connect = always_connected,
	.flags =	FLAG_NO_SETINT,

        .in = 1, .out = 2,
        //.epsize = 64,
};






static int GetVendorCommandPassive(struct mcs7830 *mcs, __u16 reg_index,__u16 length, __u8 * data)
{
        struct usb_device *dev = mcs->udev;
        int ret=0;

	//DPRINTK(" in GetVendorCommandPassive reg_index is %x   content of data %x  status %x\n",reg_index,*data,ret);
        ret = usb_control_msg(dev, usb_rcvctrlpipe(dev, 0), MCS_RD_BREQ,
                        MCS_RD_BMREQ, 0x0000, reg_index, data, length,
                        MSECS_TO_JIFFIES(MCS_CTRL_TIMEOUT));
	DPRINTK(" in GetVendorCommandPassive reg_index is %x   content of data %x  status %x\n",reg_index,*data,ret);
	if(ret <0)
	{
	DPRINTK("TRYING for 1'st time \n");
        ret = usb_control_msg(dev, usb_rcvctrlpipe(dev, 0), MCS_RD_BREQ,
                        MCS_RD_BMREQ, 0x0000, reg_index, data, length,
                        MSECS_TO_JIFFIES(MCS_CTRL_TIMEOUT));
	}
	if(ret <0)
	{
	DPRINTK("TRYING for 2'nd time \n");
        ret = usb_control_msg(dev, usb_rcvctrlpipe(dev, 0), MCS_RD_BREQ,
                        MCS_RD_BMREQ, 0x0000, reg_index, data, length,
                        MSECS_TO_JIFFIES(MCS_CTRL_TIMEOUT));
	}
	if(ret <0)
	{
	DPRINTK("TRYING for 3'rd time \n");
        ret = usb_control_msg(dev, usb_rcvctrlpipe(dev, 0), MCS_RD_BREQ,
                        MCS_RD_BMREQ, 0x0000, reg_index, data, length,
                        MSECS_TO_JIFFIES(MCS_CTRL_TIMEOUT));
	}
	if(ret <0)
	{
	DPRINTK("TRYING for 4'th time \n");
        ret = usb_control_msg(dev, usb_rcvctrlpipe(dev, 0), MCS_RD_BREQ,
                        MCS_RD_BMREQ, 0x0000, reg_index, data, length,
                        MSECS_TO_JIFFIES(MCS_CTRL_TIMEOUT));
	}
	if(ret <0)
	{
	DPRINTK("TRYING for 5'th time \n");
        ret = usb_control_msg(dev, usb_rcvctrlpipe(dev, 0), MCS_RD_BREQ,
                        MCS_RD_BMREQ, 0x0000, reg_index, data, length,
                        MSECS_TO_JIFFIES(MCS_CTRL_TIMEOUT));
	}
        return ret;
}

static int SetVendorCommandPassive(struct mcs7830 *mcs, __u16 reg_index,__u16 length, __u8 * data)
{
        struct usb_device *dev = mcs->udev;
	int ret=0;
	//DPRINTK(" in SetVendorCommandPassive reg_index is %x   content of data %x  status %x\n",reg_index,(__u16 )*data,ret);

        ret = usb_control_msg(dev, usb_sndctrlpipe(dev, 0), MCS_WR_BREQ,
                        MCS_WR_BMREQ, 0x0000, reg_index, data,length,
                        MSECS_TO_JIFFIES(MCS_CTRL_TIMEOUT));
	DPRINTK(" in SetVendorCommandPassive reg_index is %x   content of data %x  status %x\n",reg_index,*data,ret);
	if(ret <0)
	{
	DPRINTK("TRYING for 1'st time \n");
        ret = usb_control_msg(dev, usb_sndctrlpipe(dev, 0), MCS_WR_BREQ,
                        MCS_WR_BMREQ, 0x0000, reg_index, data,length,
                        MSECS_TO_JIFFIES(MCS_CTRL_TIMEOUT));
	}
	
	if(ret <0)
	{
	DPRINTK("TRYING for 2'nd time \n");
        ret = usb_control_msg(dev, usb_sndctrlpipe(dev, 0), MCS_WR_BREQ,
                        MCS_WR_BMREQ, 0x0000, reg_index, data,length,
                        MSECS_TO_JIFFIES(MCS_CTRL_TIMEOUT));
	}
	if(ret <0)
	{
	DPRINTK("TRYING for 3'rd time \n");
        ret = usb_control_msg(dev, usb_sndctrlpipe(dev, 0), MCS_WR_BREQ,
                        MCS_WR_BMREQ, 0x0000, reg_index, data,length,
                        MSECS_TO_JIFFIES(MCS_CTRL_TIMEOUT));
	}
	if(ret <0)
	{
	DPRINTK("TRYING for 4'th time \n");
        ret = usb_control_msg(dev, usb_sndctrlpipe(dev, 0), MCS_WR_BREQ,
                        MCS_WR_BMREQ, 0x0000, reg_index, data,length,
                        MSECS_TO_JIFFIES(MCS_CTRL_TIMEOUT));
	}
	if(ret <0)
	{
	DPRINTK("TRYING for 5'th time \n");
        ret = usb_control_msg(dev, usb_sndctrlpipe(dev, 0), MCS_WR_BREQ,
                        MCS_WR_BMREQ, 0x0000, reg_index, data,length,
                        MSECS_TO_JIFFIES(MCS_CTRL_TIMEOUT));
	}
	return ret;
}



static int MosUsbGetPermanentAddress(struct mcs7830 *mcs)
{
	return GetVendorCommandPassive(mcs,HIF_REG_16,ETHERNET_ADDRESS_LENGTH,mcs->AiPermanentNodeAddress);
}

static int ReadPhyRegisterPassive(struct mcs7830 *mcs,__u8 PhyRegIndex,__u8 *PhyRegValue)
{

	int status=0,Count;
	__u8 checkval=0x0;
	mcs->WordValue = 0x0;
	//__u8 checkval2[10];
	
	status=SetVendorCommandPassive(mcs,HIF_REG_11,2,&checkval);//(__u8 *)(&mcs->WordValue));
	if(status<0)
	{
		DPRINTK("ERROR: couldn't clean register HIF_REG_11&12 \n");
		return status;
	}
	else
	{
		//DPRINTK(" cleaned register HIF_REG_11&12 \n");
		checkval=0;
		
	}
	status=0;
	mcs->checkval2[0]= READ_OPCODE | FARADAY_PHY_ADDRESS;
	//DPRINTK("in Local SetVendorCommandPassive reg is %x \tval is %x\n",HIF_REG_13,mcs->checkval2[0]);// *(mcs->ByteValue));
	status=SetVendorCommandPassive(mcs,HIF_REG_13,1,&mcs->checkval2[0]);//mcs->ByteValue);
	if(status<0)
	{
		DPRINTK("ERROR :Couldn't set HIF13 for Read Mode \n");
		return status;
	}
	else
	{
		//DPRINTK("Set HIF_REG_13 with READ_OPCODE | FARADAY_PHY_ADDRESS Done!!\n");
	}

	// 2. Build value for HIF_REG14
        // Set Pend Flag, Reset Ready Flag & address of Reg.with in the PHY
        // Will procesed By MAC - MIIM Hardware.
        // Msb 1000 0000 Lsb - b7 Pend, b6 Ready,
        // b5 Not used,b4b3b2b1b0 Address of Reg with in the PHY

	status=0;
		mcs->checkval2[0]=0x0;
	mcs->checkval2[0] = HIF_REG_14_PEND_FLAG_BIT | (PhyRegIndex & 0x1F);
	status=SetVendorCommandPassive(mcs,HIF_REG_14,1,&mcs->checkval2[0]);//mcs->ByteValue);
	if(status<0)
	{
		DPRINTK("ERROR : Setting pend flag & PHY reg.index in HIF_REG_14 to read phy \n");
		return status;
	}
	else
	{
		//DPRINTK(" Setting pend flag & PHY reg.index in HIF_REG_14 to read phy Done !!!\n");
	
	}
		mcs->checkval2[0]=0x0;

	// 3. Read (completion) Status for last command
        Count = 10;
        do
        {
		status=0;
		mcs->checkval2[0]=0x0;
                status = GetVendorCommandPassive(mcs,HIF_REG_14,1,&mcs->checkval2[0]);//(__u8 *)mcs->ByteValue);
		if(status<0)
	        {
                DPRINTK("ERROR : 1Reading the ReadyFlag bit in the HIF_REG_14 \n");
		mdelay(1000);
                status = GetVendorCommandPassive(mcs,HIF_REG_14,1,&mcs->checkval2[0]);//(__u8 *)mcs->ByteValue);
		if(status < 0)
		{
                DPRINTK("ERROR : 1after Reading again the ReadyFlag bit in the HIF_REG_14 \n");
		mdelay(1000);
                status = GetVendorCommandPassive(mcs,HIF_REG_14,1,&mcs->checkval2[0]);//(__u8 *)mcs->ByteValue);
		if(status <0)
		{
                DPRINTK("ERROR : 1after Reading again 2 times the ReadyFlag bit in the HIF_REG_14 \n");
	
	                break;//command error
		}
		}
        	}
		else
		{
                //DPRINTK("Reading the ReadyFlag bit in the HIF_REG_14 DONE!!! \n");
			
		}

		//DPRINTK(" mcs->checkval2[0] value is %x\n",mcs->checkval2[0]);
		if (mcs->checkval2[0] & HIF_REG_14_READY_FLAG_BIT)
		{
			//DPRINTK("Command completed\n");
                        break; // command complete
		}
		else
		{
		
			Count -- ;
			//DPRINTK("READ PHY register command not Completed. Checking the READY flag again.\n");

			if (Count == 0 )
                        {
                 	
                                //DPRINTK("Checking the READY flag for 10 times completed not yet set.\n");
                                status = -1;
				//DPRINTK("Check1\n");
		                break;
                        }

		}
			

	}while(1);

	if(status<0)
	{
		//DPRINTK("Check2 \n");
		return status;
	}
	
	status=0;	
        status = GetVendorCommandPassive(mcs,HIF_REG_11,2,(__u8 *)PhyRegValue);
	if(status<0)
	{
		DPRINTK("ERROR :Couldn't read HIF11-12 for phy data \n");
		status=0;	
	        status = GetVendorCommandPassive(mcs,HIF_REG_11,2,(__u8 *)PhyRegValue);
		if(status < 0)
		{
		DPRINTK("ERROR :Couldn't read HIF11-12 for phy data  1 time again\n");
		status=0;	
	        status = GetVendorCommandPassive(mcs,HIF_REG_11,2,(__u8 *)PhyRegValue);
			if(status < 0)
			{		
				DPRINTK("ERROR :Couldn't read HIF11-12 for phy data  2 times again\n");

				return status;
			}
		}

	}
	return status;
}
static int WritePhyRegisterPassive(struct mcs7830 *mcs,__u8 PhyRegIndex,__u8 *PhyRegValue)
{

	int status=0,Count;
	mcs->WordValue = 0x0;
	status=SetVendorCommandPassive(mcs,HIF_REG_11,2,(__u8 *)PhyRegValue);
	if(status<0)
	{
		DPRINTK("ERROR: couldn't write data in  register HIF_REG_11&12 \n");
		return status;
	}
	else
	{
		//DPRINTK(" Wrote data to register HIF_REG_11&12 \n");
		
		
	}
	status=0;
	mcs->checkval2[0]= WRITE_OPCODE | FARADAY_PHY_ADDRESS;
	//DPRINTK("in Local SetVendorCommandPassive reg is %x \tval is %x\n",HIF_REG_13,mcs->checkval2[0]);// *(mcs->ByteValue));
	status=SetVendorCommandPassive(mcs,HIF_REG_13,1,&mcs->checkval2[0]);//mcs->ByteValue);
	if(status<0)
	{
		DPRINTK("ERROR :Couldn't set HIF13 for wRITE Mode \n");
		return status;
	}
	else
	{
		//DPRINTK("Set HIF_REG_13 with WRITE_OPCODE | FARADAY_PHY_ADDRESS Done!!\n");
	}

	// 2. Build value for HIF_REG14
        // Set Pend Flag, Reset Ready Flag & address of Reg.with in the PHY
        // Will procesed By MAC - MIIM Hardware.
        // Msb 1000 0000 Lsb - b7 Pend, b6 Ready,
        // b5 Not used,b4b3b2b1b0 Address of Reg with in the PHY

	status=0;
		mcs->checkval2[0]=0x0;
	mcs->checkval2[0] = HIF_REG_14_PEND_FLAG_BIT | (PhyRegIndex & 0x1F);
	status=SetVendorCommandPassive(mcs,HIF_REG_14,1,&mcs->checkval2[0]);//mcs->ByteValue);
	if(status<0)
	{
		DPRINTK("ERROR : Setting pend flag & PHY reg.index in HIF_REG_14 to write phy \n");
		return status;
	}
	else
	{
		//DPRINTK(" Setting pend flag & PHY reg.index in HIF_REG_14 to write phy Done !!!\n");
	
	}
		mcs->checkval2[0]=0x0;

	// 3. Read (completion) Status for last command
        Count = 10;
        do
        {
		
		status=0;
		mcs->checkval2[0]=0x0;
                status = GetVendorCommandPassive(mcs,HIF_REG_14,1,&mcs->checkval2[0]);//(__u8 *)mcs->ByteValue);
		if(status<0)
	        {
                DPRINTK("ERROR : 2Reading the ReadyFlag bit in the HIF_REG_14 \n");
		mdelay(1000);
                status = GetVendorCommandPassive(mcs,HIF_REG_14,1,&mcs->checkval2[0]);//(__u8 *)mcs->ByteValue);
		if(status < 0)
		{
                DPRINTK("ERROR : 2after Reading again the ReadyFlag bit in the HIF_REG_14 \n");
		mdelay(1000);
                status = GetVendorCommandPassive(mcs,HIF_REG_14,1,&mcs->checkval2[0]);//(__u8 *)mcs->ByteValue);
		if(status <0)
		{
                DPRINTK("ERROR : 2after Reading again 2 timesthe ReadyFlag bit in the HIF_REG_14 \n");
	
	                break;//command error
		}
		}
        	}
		else
		{
                //DPRINTK("Reading the ReadyFlag bit in the HIF_REG_14 DONE!!! \n");
			
		}

		//DPRINTK(" mcs->checkval2[0] value is %x\n",mcs->checkval2[0]);
		if (mcs->checkval2[0] & HIF_REG_14_READY_FLAG_BIT)
		{
			//DPRINTK("Command completed\n");
                        break; // command complete
		}
		else
		{
		
			Count -- ;
			//DPRINTK("READ PHY register command not Completed. Checking the READY flag again.\n");

			if (Count == 0 )
                        {
                 	
                                //DPRINTK("Checking the READY flag for 10 times completed not yet set.\n");
                                status = -1;
				//DPRINTK("Check1\n");
		                break;
                        }

		}
			

	}while(1);

	
	return status;
}


static int PrintAutoNegAdvRegister(struct mcs7830 *mcs)
{
	return 1;
}
	

static int AutoNegChangemode(struct mcs7830 *mcs,int ptrUserPhyMode)
{

	int status=0x0;
	__u8 ReadHifVal;	
	__u8 AutoNegAdvtReg_8[2];
	__u16 AutoNegAdvtReg;
	__u16 PhyControlReg_16;

	if(ptrUserPhyMode ==0)
	{
	
		// Writing 0x0021 in PhyAutoNeg Advertisement reg 4
		
		AutoNegAdvtReg = PHY_AUTONEGADVT_FdxPause | \
					PHY_AUTONEGADVT_Fdx100TX |\
                                        PHY_AUTONEGADVT_100TX | PHY_AUTONEGADVT_ieee802_3|\
                                                PHY_AUTONEGADVT_10TFdx|PHY_AUTONEGADVT_10T;
	}
	else if(ptrUserPhyMode ==1)
	{
		// Writing 0x0021 in PhyAutoNeg Advertisement reg 4
                AutoNegAdvtReg = PHY_AUTONEGADVT_10T | PHY_AUTONEGADVT_ieee802_3;
	}
	else if(ptrUserPhyMode ==2)
	{
		// Writing 0x0461 in PhyAutoNeg Advertisement reg 4
                AutoNegAdvtReg = PHY_AUTONEGADVT_FdxPause | PHY_AUTONEGADVT_10TFdx |\
                                                  PHY_AUTONEGADVT_10T | PHY_AUTONEGADVT_ieee802_3;

	}
	else if(ptrUserPhyMode ==3)
	{
		// Writing 0x0081 in PhyAutoNeg Advertisement reg 4
                AutoNegAdvtReg = PHY_AUTONEGADVT_100TX | PHY_AUTONEGADVT_ieee802_3;

	}
	else if(ptrUserPhyMode ==4)
	{
		// Writing 0x0581 in PhyAutoNeg Advertisement reg 4
                AutoNegAdvtReg = PHY_AUTONEGADVT_FdxPause | PHY_AUTONEGADVT_Fdx100TX |\
                                            PHY_AUTONEGADVT_100TX | PHY_AUTONEGADVT_ieee802_3;
	}
	//DPRINTK("in AutoNegChangemode  AutoNegAdvtReg is %04x \n", AutoNegAdvtReg);
               AutoNegAdvtReg_8[0] =  (__u8)(AutoNegAdvtReg &0xff);
               AutoNegAdvtReg_8[1] =  (__u8)((AutoNegAdvtReg >>8) &0xff);
	status=0x0;
	status=WritePhyRegisterPassive(mcs,PHY_AUTONEGADVT_REG_INDEX,AutoNegAdvtReg_8);
	//DPRINTK("Status of WritePhyRegisterPassive ANG is %x\n",status);
	if(status >=0)
        {
                //DPRINTK("Writing value for AutoNegotiation in AutoNegAdvt Register() success\n");
		//break;

        }
	else
	{	
                //DPRINTK("Writing value for AutoNegotiation in AutoNegAdvt Register() FAILURE\n");
                      // mcs7830_disconnect(mcs->udev, (void *) mcs);
		return -1;//NULL;
		}
	//////////////////Change For Link Status Problem :Begin ///////////
                      //First Disable All
		PhyControlReg_16 = 0x0000;
               mcs->PhyControlReg[0] =  (__u8)(PhyControlReg_16 &0xff);
               mcs->PhyControlReg[1] =  (__u8)((PhyControlReg_16>>8) &0xff);
	status=0x0;
               status = WritePhyRegisterPassive(mcs,PHY_CONTROL_REG_INDEX,mcs->PhyControlReg);
	if(status >=0)
               {
                       //DPRINTK("Writing value WritePhyControlRegister() success\n");
                       //break;
                }
               else
               {
                       //DPRINTK("Writing value for WritePhyControlRegister() FAILURE\n");
                       //mcs7830_disconnect(mcs->udev, (void *) mcs);
                       return -1;//NULL;
                }

		PhyControlReg_16 = 0x0000;
		PhyControlReg_16 =  PHY_CONTROL_AUTONEG_ENABLE;
               mcs->PhyControlReg[0] =  (__u8)(PhyControlReg_16 &0xff);
               mcs->PhyControlReg[1] =  (__u8)((PhyControlReg_16>>8) &0xff);
	status=0x0;
               status = WritePhyRegisterPassive(mcs,PHY_CONTROL_REG_INDEX,mcs->PhyControlReg);
	if(status >=0)
	{
        	//DPRINTK("Writing value WritePhyControlRegister() ANG enable success\n");
                //break;
        }
        else
        {
                //DPRINTK("Writing value for WritePhyControlRegister() SNG enable FAILURE\n");
                //mcs7830_disconnect(mcs->udev, (void *) mcs);
                return -1;//NULL;
        }

	//Restart Auto Neg (Keep the Enable Auto Neg Bit Set)
		PhyControlReg_16 = 0x0000;
        	PhyControlReg_16 =  PHY_CONTROL_AUTONEG_ENABLE |PHY_CONTROL_RESTART_AUTONEG;
               mcs->PhyControlReg[0] =  (__u8)(PhyControlReg_16 &0xff);
               mcs->PhyControlReg[1] =  (__u8)((PhyControlReg_16>>8) &0xff);
	status=0x0;
        status = WritePhyRegisterPassive(mcs,PHY_CONTROL_REG_INDEX,mcs->PhyControlReg);

	if(status >=0)
        {
        	//DPRINTK("Writing value WritePhyControlRegister() ANG enable&restart success\n");
                //break;

        }
        else
        {
               //DPRINTK("Writing value for WritePhyControlRegister() SNG enable&restart FAILURE\n");
               //mcs7830_disconnect(mcs->udev, (void *) mcs);
               return -1;//NULL;

        }

	mcs->bAutoNegStarted = TRUE;
	//////////////////Change For Link Status Problem :End ///////////

	
	status=0x0;
        status = ReadPhyRegisterPassive(mcs,PHY_CONTROL_REG_INDEX,mcs->TempPhyControlReg);
	if(status >=0)
        {
		//DPRINTK("PhyControl for AutoNegotiation Value: x%04x\n", mcs->TempPhyControlReg);
                //break;
        }
        else
        {
                DPRINTK("Reading value from ReadPhyControlRegister() FAILURE FAILURE\n");
                //mcs7830_disconnect(mcs->udev, (void *) mcs);
                return -1;//NULL;
        }
	// read linkspeed & duplex from PHY & set to HIF
	AutoNegAdvtReg=0x0;
	status=0x0;
        status = ReadPhyRegisterPassive(mcs,PHY_AUTONEGADVT_REG_INDEX,AutoNegAdvtReg_8);
	if(status >=0)
        {
		//DPRINTK("in AutoNegChangemode  AutoNegAdvtReg is %04x \n", AutoNegAdvtReg);
                //break;
        }
        else
        {
		//DPRINTK("in AutoNegChangemode  AutoNegAdvtReg FAILED\n");
                //mcs7830_disconnect(mcs->udev, (void *) mcs);
                return -1;//NULL;
        }

	       AutoNegAdvtReg = AutoNegAdvtReg_8[1];
	       AutoNegAdvtReg = (AutoNegAdvtReg<<8) | AutoNegAdvtReg_8[1];

          if(AutoNegAdvtReg & PHY_AUTONEGADVT_100TX) 
	 mcs->HifReg15 |=  HIF_REG_15_SPEED100; 
	else
		mcs->HifReg15 &= ~HIF_REG_15_SPEED100;


	if((AutoNegAdvtReg & PHY_AUTONEGADVT_10TFdx)||(AutoNegAdvtReg & PHY_AUTONEGADVT_Fdx100TX)) 
		 mcs->HifReg15 |= HIF_REG_15_FULLDUPLEX_ENABLE;
	else
		mcs->HifReg15 &= ~HIF_REG_15_FULLDUPLEX_ENABLE;


	status=0x0;
	status=SetVendorCommandPassive( mcs,HIF_REG_15,1,&mcs->HifReg15);
	//DPRINTK("in AutoNegChangemode  Writing mcs->HifReg15 is %04x \n",mcs->HifReg15 );
	if(status >=0)
        {
		//DPRINTK("in AutoNegChangemode  writing mcs->HifReg15 SUCCESS\n" );
                //break;
        }
        else
        {
		DPRINTK("in AutoNegChangemode  writing mcs->HifReg15 FAILED\n");
                //mcs7830_disconnect(mcs->udev, (void *) mcs);
                return -1;//NULL;
        }
	status=0;
	status=GetVendorCommandPassive( mcs,HIF_REG_15,1,&ReadHifVal);
	//DPRINTK("in AutoNegChangemode  Read mcs->HifReg15 is %04x \n",ReadHifVal );
	if(status < 0)
	{
		DPRINTK("in AutoNegChangemode  reading mcs->HifReg15 FAILED\n");
		return status;
	}

	return status;		


	//DPRINTK("Phy Status Register Value: x%04x\n", mcs->PhyStatusReg);
}

	
static int ANGInitializeDev(struct mcs7830 *mcs)
{
        int Ccnt=0;
	__u8 Read_HIFReg_22,SingleByte;
	int status;
	int i;
	mcs->DeviceReleaseNumber = DEVICE_REV_B; ///by default revision B


	do{
		status = GetVendorCommandPassive(mcs,0x15,2,&Read_HIFReg_22);
		if(status >0)//STATUS_SUCCESS)
	        {
	                DPRINTK("Device is of revision C\n");
			mcs->DeviceReleaseNumber = DEVICE_REV_C;
			break;
        	}
		Ccnt++;
		udelay(50);
	}while(Ccnt>2);

	Ccnt=5;
	do{
		status=0x0;
		status = MosUsbGetPermanentAddress(mcs);
		Ccnt--;
	}while(Ccnt>0);
	if(status >=0)
	{
                DPRINTK("vendor command MosUsbGetPermanentAddress success\n");
		DPRINTK("address is %x\n",mcs->AiPermanentNodeAddress);
		/*	
		printk("%s:address is \n:",__FUNCTION__);
		for(i=0;i<6;i++)
		printk("%x:",mcs->AiPermanentNodeAddress[i]);
		printk("\n");
		*/
	}
	

	// Check whether we have to do AutoNegotiation or the USER has forced
        // us into one mode.
        //if ( *(MosUsbContext->ptrUserPhyMode) == USER_SET_AUTONEG )
	if(1)
	{
		status=0x0;
		/*
		status=ReadPhyRegisterPassive(mcs,PHY_AUTONEGADVT_REG_INDEX,&mcs->PhyAutoNegAdvReg);
		DPRINTK("Status of ReadPhyRegisterPassive is %x\n",status);
		if(status >=0)
	        {
	                DPRINTK("ReadPhyRegisterPassive success\n");
			//break;
	
	        }
		else
		{	
	                DPRINTK("ReadPhyRegisterPassive fail\n");
                        //mcs7830_disconnect(mcs->udev, (void *) mcs);
			return -1;//NULL;

		}
		PrintAutoNegAdvRegister(mcs);
		status=0x0;
		*/
		DPRINTK("%s MODE is %d\n",__FUNCTION__,NEGOTIATE_MODE);
		status=AutoNegChangemode(mcs,NEGOTIATE_MODE);//0x05e1 org settings
		if(status>=0)
		{
			DPRINTK("Auto negotiation staarted\n");
		}
		else
		{
			DPRINTK("Auto negotiation Failed\n");
			return -1;
			
		}
		
		//DPRINTK("Phy Status Register Value: x%04x\n", mcs->PhyStatusReg);

	}




	//Rev.C change
	DPRINTK("Check1\n");	
	mcs->ptrFpgaVersion=PTR_FPGA_VERSION;
	DPRINTK("Check2\n");	
        if (mcs->ptrFpgaVersion != 0)
                mcs->HifReg15 |=  HIF_REG_15_CFG; //need to be checked!!!may be not needed in Linux driver
	DPRINTK("Check3\n");	

	// NOTE:!!!! neeed to read HifReg15 previous values!!!

    	// enable Rx & Tx register
	mcs->HifReg15 |= (HIF_REG_15_TXENABLE| HIF_REG_15_RXENABLE );

        // Temp enable Multicast bit at initialization time. It should be done
        // when driver gets the packetfilter OID.
        mcs->HifReg15 |=  HIF_REG_15_ALLMULTICAST ;
 //       mcs->HifReg15 |=  HIF_REG_15_PROMISCIOUS ;

        DPRINTK("HIF Register 15 %04x \n",mcs->HifReg15);


	DPRINTK("Check4\n");	
    	// write these values to HIF
	status=0x0;
	status=SetVendorCommandPassive( mcs,HIF_REG_15,1,&mcs->HifReg15);
        if (status>=0)
	{
        	DPRINTK (" MosUsbStartDevice() UpdateHifReg15() success\n");
	}
	else
        {
                DPRINTK("ERROR: MosUsbStartDevice() UpdateHifReg15() FAILURE (%x)\n", status);
		//mcs7830_disconnect(mcs->udev, (void *) mcs);
                return -1;//NULL;	
        }



//#ifndef MCS7830_WHQL

        //Only if we are in 10 Half mode
        if( (!(mcs->HifReg15 & HIF_REG_15_FULLDUPLEX_ENABLE)) &&
                (!(mcs->HifReg15 & HIF_REG_15_SPEED100 )) )
        {
                // Set 40 in HIF register 9
                SingleByte = 40;
                SetVendorCommandPassive( mcs,HIF_REG_09,1,&SingleByte);


                // Set 32 in HIF register 10
                SingleByte = 32;
                SetVendorCommandPassive( mcs,HIF_REG_10,1,&SingleByte);
        }


//#ifndef MCS7830_WHQL

	(mcs->ptrPauseThreshold) = PTR_PAUSE_THRESHOLD;
        if (mcs->DeviceReleaseNumber == DEVICE_REV_C)
        {
                    DPRINTK("****** PauseThreshold = %d\n", mcs->ptrPauseThreshold) ;

                // Writing pause data to the device
                SetVendorCommandPassive( mcs,HIF_REG_23,1,&mcs->ptrPauseThreshold);
        }


	return status;
}



static void mcs7830_disconnect (struct usb_device *udev, void *ptr)
{
        struct mcs7830 *mcs = (struct mcs7830 *) ptr;
	DPRINTK("mcs7830_disconnect START\n");
        //mcs->present=0;

        if (&mcs->net) 
	{
		unregister_netdev (&mcs->net);
                //mcs->net = NULL;
        }
        if (mcs->udev) 
	{

	        mcs->udev=NULL;
	}
	

        DPRINTK(" USB device MCS7830 device is removed \n");
	if(udev)
	{
        usb_put_dev (udev);
	}
	printk(" \nMOSCHIP 7830 Device Disconnected!!!!!\n");

}


#endif  /* CONFIG_USB_MOSCHIP */
/**************************MOSCHIP CODE END*********************/

#endif




/*-------------------------------------------------------------------------
 *
 * Network Device Driver (peer link to "Host Device", from USB host)
 *
 *-------------------------------------------------------------------------*/

static int mcs7830_change_mtu (struct net_device *net, int new_mtu)
{
	struct mcs7830	*dev = (struct mcs7830 *) net->priv;

	if (new_mtu <= MIN_PACKET || new_mtu > MAX_PACKET)
		return -EINVAL;
	// no second zero-length packet read wanted after mtu-sized packets
	if (((new_mtu + sizeof (struct ethhdr)) % dev->maxpacket) == 0)
		return -EDOM;
	net->mtu = new_mtu;
	return 0;
}

/*-------------------------------------------------------------------------*/

static struct net_device_stats *mcs7830_get_stats (struct net_device *net)
{
	return &((struct mcs7830 *) net->priv)->stats;
}

/*-------------------------------------------------------------------------*/

/* urb completions may be in_irq; avoid doing real work then. */

static void defer_bh (struct mcs7830 *dev, struct sk_buff *skb)
{
	struct sk_buff_head	*list = skb->list;
	unsigned long		flags;

	spin_lock_irqsave (&list->lock, flags);
	__skb_unlink (skb, list);
	spin_unlock (&list->lock);
	spin_lock (&dev->done.lock);
	__skb_queue_tail (&dev->done, skb);
	if (dev->done.qlen == 1)
		tasklet_schedule (&dev->bh);
	spin_unlock_irqrestore (&dev->done.lock, flags);
}

/* some work can't be done in tasklets, so we use keventd
 *
 * NOTE:  annoying asymmetry:  if it's active, schedule_task() fails,
 * but tasklet_schedule() doesn't.  hope the failure is rare.
 */
static void defer_kevent (struct mcs7830 *dev, int work)
{
	set_bit (work, &dev->flags);
	if (!schedule_task (&dev->kevent))
		err ("%s: kevent %d may have been dropped",
			dev->net.name, work);
	else
		dbg ("%s: kevent %d scheduled", dev->net.name, work);
}

/*-------------------------------------------------------------------------*/

static void rx_complete (struct urb *urb);

static void rx_submit (struct mcs7830 *dev, struct urb *urb, int flags)
{
	struct sk_buff		*skb;
	struct skb_data		*entry;
	int			retval = 0;
	unsigned long		lockflags;
	size_t			size;

		size = (sizeof (struct ethhdr) + dev->net.mtu);
		//Check
		size=size + PACKET_STATUS  + IP_ALIGN;
//	printk("%s size is %d \n",__FUNCTION__,size);
	if ((skb = alloc_skb (size, flags)) == 0) {
		dbg ("no rx skb");
		defer_kevent (dev, EVENT_RX_MEMORY);
		usb_free_urb (urb);
		return;
	}
	skb_reserve(skb,2);
	entry = (struct skb_data *) skb->cb;
	entry->urb = urb;
	entry->dev = dev;
	entry->state = rx_start;
	entry->length = 0;

	FILL_BULK_URB (urb, dev->udev, dev->in,
		skb->data, size, rx_complete, skb);
	urb->transfer_flags |= USB_ASYNC_UNLINK;

	spin_lock_irqsave (&dev->rxq.lock, lockflags);

	if (netif_running (&dev->net)
			&& !test_bit (EVENT_RX_HALT, &dev->flags)) {
		switch (retval = SUBMIT_URB (urb, GFP_ATOMIC)){ 
		case -EPIPE:
			defer_kevent (dev, EVENT_RX_HALT);
			break;
		case -ENOMEM:
			defer_kevent (dev, EVENT_RX_MEMORY);
			break;
		default:
			dbg ("%s rx submit, %d", dev->net.name, retval);
			tasklet_schedule (&dev->bh);
			break;
		case 0:
			__skb_queue_tail (&dev->rxq, skb);
		}
	} else {
		dbg ("rx: stopped");
		retval = -ENOLINK;
	}
	spin_unlock_irqrestore (&dev->rxq.lock, lockflags);
	if (retval) {
		dev_kfree_skb_any (skb);
		usb_free_urb (urb);
	}
}


/*-------------------------------------------------------------------------*/

static inline void rx_process (struct mcs7830 *dev, struct sk_buff *skb)
{
	if (dev->driver_info->rx_fixup
			&& !dev->driver_info->rx_fixup (dev, skb))
		goto error;
	// else network stack removes extra byte if we forced a short packet

	if (skb->len) {
		int	status;

		skb->dev = &dev->net;
		skb->protocol = eth_type_trans (skb, &dev->net);
		dev->stats.rx_packets++;
		dev->stats.rx_bytes += skb->len;

#ifdef	VERBOSE
		devdbg (dev, "< rx, len %d, type 0x%x",
			skb->len + sizeof (struct ethhdr), skb->protocol);
#endif
		memset (skb->cb, 0, sizeof (struct skb_data));
		status = netif_rx (skb);
		if (status != NET_RX_SUCCESS)
			devdbg (dev, "netif_rx status %d", status);
	} else {
		dbg ("drop");
error:
		dev->stats.rx_errors++;
		skb_queue_tail (&dev->done, skb);
	}
}

/*-------------------------------------------------------------------------*/

static void rx_complete (struct urb *urb)
{
	struct sk_buff		*skb = (struct sk_buff *) urb->context;
	struct skb_data		*entry = (struct skb_data *) skb->cb;
	struct mcs7830		*dev = entry->dev;
	int			urb_status = urb->status;

	skb_put (skb, urb->actual_length);
	entry->state = rx_done;
	entry->urb = 0;

	switch (urb_status) {
	    // success
	    case 0:
		if (MIN_PACKET > skb->len || skb->len > MAX_PACKET) {
			entry->state = rx_cleanup;
			dev->stats.rx_errors++;
			dev->stats.rx_length_errors++;
			dbg ("rx length %d", skb->len);
		}
		break;

	    // stalls need manual reset. this is rare ... except that
	    // when going through USB 2.0 TTs, unplug appears this way.
	    // we avoid the highspeed version of the ETIMEOUT/EILSEQ
	    // storm, recovering as needed.
	    case -EPIPE:
		defer_kevent (dev, EVENT_RX_HALT);
		// FALLTHROUGH

	    // software-driven interface shutdown
	    case -ECONNRESET:		// according to API spec
	    case -ECONNABORTED:		// some (now fixed?) UHCI bugs
		dbg ("%s rx shutdown, code %d", dev->net.name, urb_status);
		entry->state = rx_cleanup;
		// do urb frees only in the tasklet (UHCI has oopsed ...)
		entry->urb = urb;
		urb = 0;
		break;

	    // data overrun ... flush fifo?
	    case -EOVERFLOW:
		dev->stats.rx_over_errors++;
		// FALLTHROUGH
	    
	    default:
		// on unplug we get ETIMEDOUT (ohci) or EILSEQ (uhci)
		// until khubd sees its interrupt and disconnects us.
		// that can easily be hundreds of passes through here.
		entry->state = rx_cleanup;
		dev->stats.rx_errors++;
		dbg ("%s rx: status %d", dev->net.name, urb_status);
		break;
	}

	defer_bh (dev, skb);

	if (urb) {
		if (netif_running (&dev->net)
				&& !test_bit (EVENT_RX_HALT, &dev->flags)) {
			rx_submit (dev, urb, GFP_ATOMIC);
			return;
		}
		usb_free_urb (urb);
	}
#ifdef	VERBOSE
	dbg ("no read resubmitted");
#endif /* VERBOSE */
}

/*-------------------------------------------------------------------------*/

// unlink pending rx/tx; completion handlers do all other cleanup

static int unlink_urbs (struct sk_buff_head *q)
{
	unsigned long		flags;
	struct sk_buff		*skb, *skbnext;
	int			count = 0;

	spin_lock_irqsave (&q->lock, flags);
	for (skb = q->next; skb != (struct sk_buff *) q; skb = skbnext) {
		struct skb_data		*entry;
		struct urb		*urb;
		int			retval;

		entry = (struct skb_data *) skb->cb;
		urb = entry->urb;
		skbnext = skb->next;

		// during some PM-driven resume scenarios,
		// these (async) unlinks complete immediately
		retval = usb_unlink_urb (urb);
		if (retval != -EINPROGRESS && retval != 0)
			dbg ("unlink urb err, %d", retval);
		else
			count++;
	}
	spin_unlock_irqrestore (&q->lock, flags);
	return count;
}


/*-------------------------------------------------------------------------*/

// precondition: never called in_interrupt

static int mcs7830_stop (struct net_device *net)
{
	struct mcs7830		*dev = (struct mcs7830 *) net->priv;
	int			temp;
	DECLARE_WAIT_QUEUE_HEAD (unlink_wakeup); 
	DECLARE_WAITQUEUE (wait, current);

	mutex_lock (&dev->mutex);
	netif_stop_queue (net);

	if (dev->msg_level >= 2)
		devinfo (dev, "stop stats: rx/tx %ld/%ld, errs %ld/%ld",
			dev->stats.rx_packets, dev->stats.tx_packets, 
			dev->stats.rx_errors, dev->stats.tx_errors
			);

	// ensure there are no more active urbs
	add_wait_queue (&unlink_wakeup, &wait);
	dev->wait = &unlink_wakeup;
	temp = unlink_urbs (&dev->txq) + unlink_urbs (&dev->rxq);

	// maybe wait for deletions to finish.
	while (skb_queue_len (&dev->rxq)
			&& skb_queue_len (&dev->txq)
			&& skb_queue_len (&dev->done)) {
		set_current_state (TASK_UNINTERRUPTIBLE);
		schedule_timeout (UNLINK_TIMEOUT_JIFFIES);
		dbg ("waited for %d urb completions", temp);
	}
	dev->wait = 0;
	remove_wait_queue (&unlink_wakeup, &wait); 

	mutex_unlock (&dev->mutex);
	return 0;
}

/*-------------------------------------------------------------------------*/

// posts reads, and enables write queing

// precondition: never called in_interrupt

static int mcs7830_open (struct net_device *net)
{
	struct mcs7830		*dev = (struct mcs7830 *) net->priv;
	int			retval = 0;
	struct driver_info	*info = dev->driver_info;

	mutex_lock (&dev->mutex);

	// put into "known safe" state
	if (info->reset && (retval = info->reset (dev)) < 0) {
		devinfo (dev, "open reset fail (%d) mcs7830 usb- %s",
			retval,
			//dev->udev->bus->bus_name, dev->udev->devpath,
			info->description);
		goto done;
	}

	// insist peer be connected
	if (info->check_connect && (retval = info->check_connect (dev)) < 0) {
		devdbg (dev, "can't open; %d", retval);
		goto done;
	}

	netif_start_queue (net);

	// delay posting reads until we're fully open
	tasklet_schedule (&dev->bh);
done:
	mutex_unlock (&dev->mutex);
	return retval;
}

/*-------------------------------------------------------------------------*/
/*
static void mcs7830_get_drvinfo (struct net_device *net, struct ethtool_drvinfo *info)
{
	struct mcs7830 *dev = net->priv;

	strncpy (info->driver, driver_name, sizeof info->driver);
	strncpy (info->version, DRIVER_VERSION, sizeof info->version);
	strncpy (info->fw_version, dev->driver_info->description,
		sizeof info->fw_version);
	usb_make_path (dev->udev, info->bus_info, sizeof info->bus_info);
}
*/
static u32 mcs7830_get_link (struct net_device *net)
{
	struct mcs7830 *dev = net->priv;

	/* If a check_connect is defined, return it's results */
	if (dev->driver_info->check_connect)
		return dev->driver_info->check_connect (dev) == 0;

	/* Otherwise, we're up to avoid breaking scripts */
	return 1;
}

static u32 mcs7830_get_msglevel (struct net_device *net)
{
	struct mcs7830 *dev = net->priv;

	return dev->msg_level;
}

static void mcs7830_set_msglevel (struct net_device *net, u32 level)
{
	struct mcs7830 *dev = net->priv;

	dev->msg_level = level;
}

static int mcs7830_ioctl (struct net_device *net, struct ifreq *rq, int cmd)
{
#ifdef NEED_MII
	{
	struct mcs7830 *dev = (struct mcs7830 *)net->priv;

	if (dev->mii.mdio_read != NULL && dev->mii.mdio_write != NULL)
		/*return generic_mii_ioctl(&dev->mii,
				(struct mii_ioctl_data *) &rq->ifr_data,
				cmd, NULL);*/
			return 0;
	}
#endif
	return -EOPNOTSUPP;
}

/*-------------------------------------------------------------------------*/

/* work that cannot be done in interrupt context uses keventd.
 *
 * NOTE:  "uhci" and "usb-uhci" may have trouble with this since they don't
 * queue control transfers to individual devices, and other threads could
 * trigger control requests concurrently.  hope that's rare.
 */
static void
kevent (void *data)
{
	struct mcs7830		*dev = data;
	int			status;

	/* usb_clear_halt() needs a thread context */
	if (test_bit (EVENT_TX_HALT, &dev->flags)) {
		unlink_urbs (&dev->txq);
		status = usb_clear_halt (dev->udev, dev->out);
		if (status < 0)
			err ("%s: can't clear tx halt, status %d",
				dev->net.name, status);
		else {
			clear_bit (EVENT_TX_HALT, &dev->flags);
			netif_wake_queue (&dev->net);
		}
	}
	if (test_bit (EVENT_RX_HALT, &dev->flags)) {
		unlink_urbs (&dev->rxq);
		status = usb_clear_halt (dev->udev, dev->in);
		if (status < 0)
			err ("%s: can't clear rx halt, status %d",
				dev->net.name, status);
		else {
			clear_bit (EVENT_RX_HALT, &dev->flags);
			tasklet_schedule (&dev->bh);
		}
	}

	/* tasklet could resubmit itself forever if memory is tight */
	if (test_bit (EVENT_RX_MEMORY, &dev->flags)) {
		struct urb	*urb = 0;

		if (netif_running (&dev->net))
			urb = ALLOC_URB (0, GFP_KERNEL);
		else
			clear_bit (EVENT_RX_MEMORY, &dev->flags);
		if (urb != 0) {
			clear_bit (EVENT_RX_MEMORY, &dev->flags);
			rx_submit (dev, urb, GFP_KERNEL);
			tasklet_schedule (&dev->bh);
		}
	}

	if (dev->flags)
		dbg ("%s: kevent done, flags = 0x%lx",
			dev->net.name, dev->flags);
}

/*-------------------------------------------------------------------------*/

static void tx_complete (struct urb *urb)
{
	struct sk_buff		*skb = (struct sk_buff *) urb->context;
	struct skb_data		*entry = (struct skb_data *) skb->cb;
	struct mcs7830		*dev = entry->dev;

	if (urb->status == -EPIPE)
		defer_kevent (dev, EVENT_TX_HALT);
	urb->dev = 0;
	entry->state = tx_done;
	defer_bh (dev, skb);
}

/*-------------------------------------------------------------------------*/

static void mcs7830_tx_timeout (struct net_device *net)
{
	struct mcs7830		*dev = (struct mcs7830 *) net->priv;

	unlink_urbs (&dev->txq);
	tasklet_schedule (&dev->bh);

	// FIXME: device recovery -- reset?
}

/*-------------------------------------------------------------------------*/

static int mcs7830_start_xmit (struct sk_buff *skb, struct net_device *net)
{
	struct mcs7830		*dev = (struct mcs7830 *) net->priv;
	int			length;
	int			retval = NET_XMIT_SUCCESS;
	struct urb		*urb = 0;
	struct skb_data		*entry;
	struct driver_info	*info = dev->driver_info;
	unsigned long		flags;

	// some devices want funky USB-level framing, for
	// win32 driver (usually) and/or hardware quirks
	if (info->tx_fixup) {
		skb = info->tx_fixup (dev, skb, GFP_ATOMIC);
		if (!skb) {
			dbg ("can't tx_fixup skb");
			goto drop;
		}
	}
	length = skb->len;

	if (!(urb = ALLOC_URB (0, GFP_ATOMIC))) {
		dbg ("no urb");
		goto drop;
	}

	entry = (struct skb_data *) skb->cb;
	entry->urb = urb;
	entry->dev = dev;
	entry->state = tx_start;
	entry->length = length;

	// FIXME: reorganize a bit, so that fixup() fills out NetChip
	// framing too. (Packet ID update needs the spinlock...)
	// [ BETTER:  we already own net->xmit_lock, that's enough ]


	/* don't assume the hardware handles USB_ZERO_PACKET */
	if ((length % dev->maxpacket) == 0)
		skb->len++;

	FILL_BULK_URB (urb, dev->udev, dev->out,
			skb->data, skb->len, tx_complete, skb);
	urb->transfer_flags |= USB_ASYNC_UNLINK;

	spin_lock_irqsave (&dev->txq.lock, flags);


	switch ((retval = SUBMIT_URB (urb, GFP_ATOMIC))) {
	case -EPIPE:
		netif_stop_queue (net);
		defer_kevent (dev, EVENT_TX_HALT);
		break;
	default:
		dbg ("%s tx: submit urb err %d", net->name, retval);
		break;
	case 0:
		net->trans_start = jiffies;
		__skb_queue_tail (&dev->txq, skb);
		if (dev->txq.qlen >= TX_QLEN)
			netif_stop_queue (net);
	}
	spin_unlock_irqrestore (&dev->txq.lock, flags);

	if (retval) {
		devdbg (dev, "drop, code %d", retval);
drop:
		retval = NET_XMIT_SUCCESS;
		dev->stats.tx_dropped++;
		if (skb)
			dev_kfree_skb_any (skb);
		usb_free_urb (urb);
#ifdef	VERBOSE
	} else {
		devdbg (dev, "> tx, len %d, type 0x%x",
			length, skb->protocol);
#endif
	}
	return retval;
}


/*-------------------------------------------------------------------------*/

// tasklet ... work that avoided running in_irq()

static void mcs7830_bh (unsigned long param)
{
	struct mcs7830		*dev = (struct mcs7830 *) param;
	struct sk_buff		*skb;
	struct skb_data		*entry;

	while ((skb = skb_dequeue (&dev->done))) {
		entry = (struct skb_data *) skb->cb;
		switch (entry->state) {
		    case rx_done:
			entry->state = rx_cleanup;
			rx_process (dev, skb);
			continue;
		    case tx_done:
			if (entry->urb->status) {
				// can this statistic become more specific?
				dev->stats.tx_errors++;
				dbg ("%s tx: err %d", dev->net.name,
					entry->urb->status);
			} else {
				dev->stats.tx_packets++;
				dev->stats.tx_bytes += entry->length;
			}
			// FALLTHROUGH:
		    case rx_cleanup:
			usb_free_urb (entry->urb);
			dev_kfree_skb (skb);
			continue;
		    default:
			dbg ("%s: bogus skb state %d",
				dev->net.name, entry->state);
		}
	}

	// waiting for all pending urbs to complete?
	if (dev->wait) {
		if ((dev->txq.qlen + dev->rxq.qlen + dev->done.qlen) == 0) {
			wake_up (dev->wait);
		}

	// or are we maybe short a few urbs?
	} else if (netif_running (&dev->net)
			&& !test_bit (EVENT_RX_HALT, &dev->flags)) {
		int	temp = dev->rxq.qlen;

		if (temp < RX_QLEN) {
			struct urb	*urb;
			int		i;
			for (i = 0; i < 3 && dev->rxq.qlen < RX_QLEN; i++) {
				if ((urb = ALLOC_URB (0, GFP_ATOMIC)) != 0)
					rx_submit (dev, urb, GFP_ATOMIC);
			}
			if (temp != dev->rxq.qlen)
				devdbg (dev, "rxqlen %d --> %d",
						temp, dev->rxq.qlen);
			if (dev->rxq.qlen < RX_QLEN)
				tasklet_schedule (&dev->bh);
		}
		if (dev->txq.qlen < TX_QLEN)
			netif_wake_queue (&dev->net);
	}
}


#ifdef USBNET
/*-------------------------------------------------------------------------
 *
 * USB Device Driver support
 *
 *-------------------------------------------------------------------------*/
 
// precondition: never called in_interrupt

static void usbnet_disconnect (struct usb_device *udev, void *ptr)
{
	struct usbnet	*dev = (struct usbnet *) ptr;

	devinfo (dev, "unregister usbnet usb- %s",
		//udev->bus->bus_name, udev->devpath,
		dev->driver_info->description);
	
	unregister_netdev (&dev->net);

	mutex_lock (&usbnet_mutex);
	mutex_lock (&dev->mutex);
	list_del (&dev->dev_list);
	mutex_unlock (&usbnet_mutex);

	// assuming we used keventd, it must quiesce too
	flush_scheduled_tasks ();

	kfree (dev);
	usb_put_dev (udev);
}

#endif
/*-------------------------------------------------------------------------*/

// precondition: never called in_interrupt

static void *
mcs7830_probe (struct usb_device *udev, unsigned ifnum,
			const struct usb_device_id *prod)
{
	struct mcs7830			*dev;
	struct net_device 		*net;
	struct driver_info		*info;
	int				altnum = 0;
	int				status;
	int i;	
	int ret;
	info = (struct driver_info *) prod->driver_info;
	printk("\nMCS7830 :Please wait for device to complete initialization!!!!!\n");

	// more sanity (unless the device is broken)
	if (!(info->flags & FLAG_NO_SETINT)) {
		if (usb_set_interface (udev, ifnum, altnum) < 0) {
			err ("set_interface failed");
			return 0;
		}
	}

	// set up our own records
	if (!(dev = kmalloc (sizeof *dev, GFP_KERNEL))) {
		dbg ("can't kmalloc dev");
		return 0;
	}
	memset (dev, 0, sizeof *dev);

	printk("%s: size of device is %d\n",__FUNCTION__,sizeof *dev);
	
	init_MUTEX_LOCKED (&dev->mutex);
	usb_get_dev (udev);
	dev->udev = udev;
	dev->driver_info = info;
	dev->msg_level = msg_level;
        if(udev->descriptor.idVendor == 0x9710 &&  udev->descriptor.idProduct == 0x7830 )
	{	
		DPRINTK("MOSCHIP 7830 connected\n");
		dev->is_7830_connected=1;

	}
		
	if(dev->is_7830_connected==1)
	{
#ifdef CONFIG_USB_MOSCHIP
//        if(udev->descriptor.idVendor == 0x9710 &&  udev->descriptor.idProduct == 0x7830 )

	if((dev)->udev->speed == USB_SPEED_HIGH) DPRINTK("device is USB2.0\n"); 
	else
	DPRINTK("device is USB1.1\n");
        {
        usb_get_dev (udev);

        DPRINTK(" USB device MCS7830 device is attached \n");
        }
        status=0;
        status=ANGInitializeDev(dev);
        if(status >=0)
	{
	        DPRINTK("Device initialization is completed\n");
	}
        else
        {
                DPRINTK("ERROR: device initialization  FAILURE (%x)\n", status);
		
		//if(no_of_times_probe_failed > 3)
		{
                printk("ERROR:Device initialization FAILED!! reconnect Device\n");
	                mcs7830_disconnect(dev->udev, (void *) dev);
			no_of_times_probe_failed= 0;

        	        return 0;//-ENODEV;//-1;//NULL;
		}
		//	no_of_times_probe_failed++;
		//	return mcs7830_probe(udev,ifnum,prod);
		
        }
		/*

		printk("%s:address is \n:",__FUNCTION__);
		for(i=0;i<6;i++)
		printk("%x:",dev->AiPermanentNodeAddress[i]);
		printk("\n");
		*/	
	/*	
	dev->mii.dev = net;
        dev->mii.mdio_read = mdio_read;
        dev->mii.mdio_write = mdio_write;
	dev->mii.phy_id_mask = 0x3f;
        dev->mii.reg_num_mask = 0x1f;
	dev->mii.phy_id = *((u8 *)dev->AiPermanentNodeAddress + 1);
	*/
	
#endif
	}
	
	INIT_LIST_HEAD (&dev->dev_list);
	skb_queue_head_init (&dev->rxq);
	skb_queue_head_init (&dev->txq);
	skb_queue_head_init (&dev->done);
	dev->bh.func = mcs7830_bh;
	dev->bh.data = (unsigned long) dev;
	INIT_TQUEUE (&dev->kevent, kevent, dev);

	// set up network interface records
	net = &dev->net;
	SET_MODULE_OWNER (net);
	net->priv = dev;
	strcpy (net->name, "usb%d");
	//memcpy (net->dev_addr, node_id, sizeof node_id);

        for (i = 0; i < 6; i++) // Hardware Address
        {
                net->dev_addr[i] = dev->AiPermanentNodeAddress[i];
                net->broadcast[i] = 0xff;
        }
	// point-to-point link ... we always use Ethernet headers 
	// supports win32 interop and the bridge driver.
	ether_setup (net);

	net->change_mtu = mcs7830_change_mtu;
	net->get_stats = mcs7830_get_stats;
	net->hard_start_xmit = mcs7830_start_xmit;
	net->open = mcs7830_open;
	net->stop = mcs7830_stop;
	net->watchdog_timeo = TX_TIMEOUT_JIFFIES;
	net->tx_timeout = mcs7830_tx_timeout;
	net->do_ioctl = mcs7830_ioctl;
	//net->ethtool_ops = &mcs7830_ethtool_ops;





	// allow device-specific bind/init procedures
	// NOTE net->name still not usable ...
	if (info->bind) {
		status = info->bind (dev, udev);
		// heuristic:  "usb%d" for links we know are two-host,
		// else "eth%d" when there's reasonable doubt.  userspace
		// can rename the link if it knows better.
		if ((dev->driver_info->flags & FLAG_ETHER) != 0
				&& (net->dev_addr [0] & 0x02) == 0)
			strcpy (net->name, "eth%d");
	} else if (!info->in || info->out)
		status = get_endpoints (dev, udev->actconfig->interface + ifnum);
	else {
		dev->in = usb_rcvbulkpipe (udev, info->in);
		dev->out = usb_sndbulkpipe (udev, info->out);
	}

	dev->maxpacket = usb_maxpacket (dev->udev, dev->out, 1);

	register_netdev (&dev->net);
	devinfo (dev, "register mcs7830 usb- %s",
		//udev->bus->bus_name, udev->devpath,
		dev->driver_info->description);

	// ok, it's ready to go.
	mutex_lock (&mcs7830_mutex);
	list_add (&dev->dev_list, &mcs7830_list);
	mutex_unlock (&dev->mutex);

	// start as if the link is up
	netif_device_attach (&dev->net);

	mutex_unlock (&mcs7830_mutex);
	printk("%s :Device initialization is completed\n",net->name);
	//printk("Ready to USE the device %s !!\n",net->name);
	return dev;
}


/*-------------------------------------------------------------------------*/

/*
 * chip vendor names won't normally be on the cables, and
 * may not be on the device.
 */

static const struct usb_device_id	products [] = {


#ifdef  CONFIG_USB_MOSCHIP
{
        USB_DEVICE (MCS_VENDOR_ID,MCS_PRODUCT_ID),      //MOSCHIP7830
        .driver_info =  (unsigned long) &moschip_info,
},
#endif



	{ },		// END
};
MODULE_DEVICE_TABLE (usb, products);

static struct usb_driver mcs7830_driver = {
	.name =		driver_name,
	.id_table =	products,
	.probe =	mcs7830_probe,
	.disconnect =	mcs7830_disconnect,
};
#ifdef ETHtool
/* Default ethtool_ops assigned.  Devices can override in their bind() routine */
static struct ethtool_ops mcs7830_ethtool_ops = {
	//.get_drvinfo		= mcs7830_get_drvinfo,
	.get_drvinfo		= NULL,
	.get_link		= mcs7830_get_link,
	.get_msglevel		= mcs7830_get_msglevel,
	.set_msglevel		= mcs7830_set_msglevel,
};
#endif
/*-------------------------------------------------------------------------*/

static int __init mcs7830_init (void)
{
	// compiler should optimize this out
	if (sizeof (((struct sk_buff *)0)->cb) < sizeof (struct skb_data))
		BUG ();

	get_random_bytes (node_id, sizeof node_id);
	node_id [0] &= 0xfe;	// clear multicast bit
	node_id [0] |= 0x02;    // set local assignment bit (IEEE802)

 	if (usb_register (&mcs7830_driver) < 0)
 		return -1;

	return 0;
}
module_init (mcs7830_init);

static void __exit mcs7830_exit (void)
{
 	usb_deregister (&mcs7830_driver);
}
module_exit (mcs7830_exit);

EXPORT_NO_SYMBOLS;
MODULE_DESCRIPTION ("moschip usb network adapter");
MODULE_LICENSE ("GPL");
